"use client";
import { useState, useEffect } from "react";
import { Loader2 } from "lucide-react";
import { useLanguage } from "@/contexts/LanguageContext";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";

interface Technician {
  id: string;
  name?: string | null;
  email: string;
  hourlyRate?: number | null;
  isExternal?: boolean;
  companyName?: string | null;
  customRole?: {
    id: string;
    name: string;
  } | null;
}

interface Role {
  id: string;
  name: string;
}

interface EditTechnicianDialogProps {
  isOpen: boolean;
  technician: Technician | null;
  roles: Role[];
  onClose: () => void;
  onSave: (data: {
    name: string;
    email: string;
    hourlyRate: number | null;
    customRoleId: string | null;
    isExternal: boolean;
    companyName: string | null;
  }) => Promise<void>;
}

export default function EditTechnicianDialog({
  isOpen,
  technician,
  roles,
  onClose,
  onSave,
}: EditTechnicianDialogProps) {
  const { t } = useLanguage();
  const [name, setName] = useState("");
  const [email, setEmail] = useState("");
  const [hourlyRate, setHourlyRate] = useState("");
  const [roleId, setRoleId] = useState("");
  const [isExternal, setIsExternal] = useState(false);
  const [companyName, setCompanyName] = useState("");
  const [saving, setSaving] = useState(false);

  // Reset form when technician changes
  useEffect(() => {
    if (technician) {
      setName(technician.name || "");
      setEmail(technician.email || "");
      setHourlyRate(technician.hourlyRate?.toString() || "");
      setRoleId(technician.customRole?.id || "");
      setIsExternal(technician.isExternal || false);
      setCompanyName(technician.companyName || "");
    }
  }, [technician]);

  const handleSave = async () => {
    if (!technician) return;

    setSaving(true);
    try {
      await onSave({
        name: name || "",
        email: email,
        hourlyRate: hourlyRate ? parseFloat(hourlyRate) : null,
        customRoleId: roleId || null,
        isExternal: isExternal,
        companyName: companyName || null,
      });
      onClose();
    } catch (error) {
      // Error handling is done in parent component
    } finally {
      setSaving(false);
    }
  };

  if (!technician || !isOpen) return null;

  return (
    <Dialog
      open={isOpen}
      onOpenChange={(open) => !open && onClose()}
      modal={true}
    >
      <DialogContent className='sm:max-w-[600px]'>
        <DialogHeader>
          <DialogTitle>{t("admin.editTechnician")}</DialogTitle>
        </DialogHeader>

        <div className='grid gap-4 py-4 px-1 max-h-[50vh] overflow-y-auto pb-12'>
          <div className='grid gap-2'>
            <Label htmlFor='name'>{t("createTicket.anonymous.name")}</Label>
            <Input
              id='name'
              type='text'
              value={name}
              onChange={(e) => setName(e.target.value)}
              placeholder={t("createTicket.anonymous.namePlaceholder")}
            />
          </div>

          <div className='grid gap-2'>
            <Label htmlFor='email'>{t("login.email")}</Label>
            <Input
              id='email'
              type='email'
              value={email}
              onChange={(e) => setEmail(e.target.value)}
              placeholder={t("login.enterEmail")}
            />
          </div>

          <div className='grid gap-2'>
            <Label htmlFor='hourlyRate'>Hourly Rate (€/hour)</Label>
            <Input
              id='hourlyRate'
              type='number'
              value={hourlyRate}
              onChange={(e) => setHourlyRate(e.target.value)}
              placeholder='25.00'
              step='0.01'
              min='0'
            />
          </div>

          <div className='grid gap-2'>
            <Label htmlFor='role'>{t("sidebar.roles")}</Label>
            <Select
              value={roleId || "none"}
              onValueChange={(value) =>
                setRoleId(value === "none" ? "" : value)
              }
            >
              <SelectTrigger id='role'>
                <SelectValue placeholder={t("admin.noRole")} />
              </SelectTrigger>
              <SelectContent
                position='popper'
                sideOffset={4}
                className='max-h-[200px] overflow-y-auto'
              >
                <SelectItem value='none'>{t("admin.noRole")}</SelectItem>
                {roles.map((role) => (
                  <SelectItem key={role.id} value={role.id}>
                    {role.name}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          <div className='grid gap-2'>
            <Label htmlFor='isExternal' className='flex items-center gap-2'>
              <input
                type='checkbox'
                id='isExternal'
                checked={isExternal}
                onChange={(e) => setIsExternal(e.target.checked)}
                className='w-4 h-4'
              />
              External Maintenance Company
            </Label>
          </div>

          {isExternal && (
            <div className='grid gap-2'>
              <Label htmlFor='companyName'>Company Name</Label>
              <Input
                id='companyName'
                type='text'
                value={companyName}
                onChange={(e) => setCompanyName(e.target.value)}
                placeholder='Company name'
              />
            </div>
          )}
        </div>

        <div className='flex justify-end gap-3'>
          <Button variant='outline' onClick={onClose}>
            {t("common.cancel")}
          </Button>
          <Button onClick={handleSave} disabled={saving}>
            {saving ? (
              <>
                <Loader2 className='mr-2 h-4 w-4 animate-spin' />
                {t("admin.savingChanges")}
              </>
            ) : (
              t("admin.saveChanges")
            )}
          </Button>
        </div>
      </DialogContent>
    </Dialog>
  );
}
